/**
 *
 * \file        ir_receiver.h
 *
 * \brief       RC5 IR Receiver class Definitions
 *
 * \author      Dennis Hromin
 *
 * \date        12/16/2008
 *
 */
#ifndef _IRRECEIVER_H_
#define _IRRECEIVER_H_
#include "nutypedefs.h"


class  IrReceiver;
extern IrReceiver * g_IrReceiver;   //global pointer to IrReceiver Class

#ifndef TRUE
#define TRUE  1
#endif

#ifndef FALSE
#define FALSE 0
#endif

#define IR_DECODE_FAILURE	0
#define IR_DECODE_SUCCESS	1
#define IR_DECODE_COMPLETE	2

// parameters for Transmitter tasks and queues
#define IR_RC5_QUEUE_LENGTH         8  // Number can be reduced if needed. Uses total of 16 bytes in queue.
#define IR_RC5_TASK_PRIORITY        0  //No Priorities setup currently, default is 0
#define IR_RC5_TASK_STACK           (32/sizeof(UINT32))

//	RC5 Specific Packet Defines
#define	RC5_MAX_BITS				14	   // No of RC5 bits
#define RC5_ADDR_MASK				0x1F   //5 bits for the addr mask
#define RC5_DATA_MASK				0x3F   //6 bits for the data mask
#define RC5_ADDR_BITS				0x5   //total 5 bits for the address
#define RC5_DATA_BITS				0x6   //total 6 bits for the data

#define RC5_CODE_TIMEOUT			150	  // 150 msec	<allow for ~ 1-packet gap time>
#define RC5_CODE_NULL				0	  //Special code to indicate to main IR task when to re-trigger the timeout timer

#define RC5_CODE_STATE_RELEASE 	0	//indicates this is the released IR code state
#define RC5_CODE_STATE_ACTIVE 	1	//indicates this is the active processed IR RC5 code state

#define RC5_CODE_TIMEOUT_INTERVAL_DELAY		10	//delay task in units of max 10 ms

/*--- IMPORTANT NOTE:  Timer Clock Rate for Input Capture/Output Compare MUST be set to 250 KHz ---*/
//	RC5 Bit Time thresholds:
//==========================
//	standard RC5 has bit time of 1.78msec and one-half bit time of 889usec.
//	the crestron third party remotes bit times are faster by a factor of 38/36=1.055
//	the bit time for the faster remotes is 1.684msec and half-bit time of 842usec.
//	the bit and half-bit min threshold is based on the faster remote bit timing of 20% and 40%
//	the bit and half-bit max threshold is based on the standard remote bit timing of 20% and 40%
#define	RC5_HALF_BIT_TIME_MIN		((842-336)/4)				//	(1.684msec / 2) - 40%
#define	RC5_HALF_BIT_TIME_MAX		((889+355)/4)				//	(1.78msec / 2)  + 40%
#define	RC5_BIT_TIME_MIN			((1684-336)/4)				//	1.684msec - 20%
#define	RC5_BIT_TIME_MAX			((1788+355)/4)				//	1.78msec + 20%
#define	RC5_PKT_TIMEOUT				(RC5_BIT_TIME_MAX+(133/4))	//	RC5 code timeout

//=================================================================
// RC5 codes mask
// Bit0..Bit5  = Command code
// Bit6..Bit10 = ID code
// Bit11       = Toggle bit
// Bit12..Bit13= Start bits
//=================================================================
#define	RC5_START_BIT_MASK_VER1		0x3000
#define	RC5_START_BIT_MASK_VER2		0x2000
#define	RC5_TOGGLE_BIT_MASK			0x0800
#define	RC5_ID_CODE_MASK			0x07C0
#define	RC5_CMD_CODE_MASK			0x003F


// RC5 code ISR Decoding variables/states
typedef	struct sIrRC5PacketStatus
{
	UINT8   IRCodeState;  //indicates which IR Code State we are in, usually Active or Release Code
	UINT16	Code;   //stores the current Final RC5 decoded code
	UINT16	PrevCode; //stores the prev Final RC5 decoded code
	UINT16	TimeOut; //stores the decoding timeout count
	UINT16	TmpCode; //temp code stored during the decode process

	// Variables used during decode process in RC5 ISR //
	UINT32	SampleTime;
	UINT32	NextSampleEdge;
	UINT32	SampleAcc;
	UINT32	SampleAccImg;
	UINT8	SampleHalfBitsCnt;

}RC5_TYPE_STRUCT;

/* Struct that stores the 3 data bytes or received RC5 code to report back via cresnet to ctrl system */
typedef	struct sIrRC5PacketTransmit
{
	UINT8 data0;
	UINT8 data1;
	UINT8 data2;
}RF1WAY_TYPE_STRUCT;


// The IR Receiver class is a generic reciever for decoding incoming IR RC5 pulses only.  It runs as a task
// which periodically wakes up to monitor the RC5 queue for any new packets to decode and send to ctrl system
class IrReceiver
{
public:
    // create object with its configuration and assign it to its stream
	IrReceiver(void);  //The constructor func definition
    void RC5_IrProcessTask(void); // Setup the main IR Process task
	void RC5_RestartCapture(void); //re-inits the rc5 capture isr routine
	void RC5_ProcessTimerISR (void); //calls ISR function to process the TIMx ISR handler

    /* Functions for IRInput To Be Used for TP Docking Station Application */
    void IrDock_RestartCapture(void); //re-inits the capture isr routine
    void IrDock_ProcessTimerISR(void); 
    void IrDock_TreatIrInputAsDock(void)
	{
		m_Dock = TRUE;
	}

    RC5_TYPE_STRUCT m_RC5; //member variable defined Struct that has current Ir Packet Information
	RF1WAY_TYPE_STRUCT m_RF1way; //member variable defined struct for sending out 1way type 0x09 RF packet (for RC5)

private:
	/* Private functions local to IR Reciever class */
	BOOL RC5_ProcessInputCapture(UINT16 CaptureTime);
	BOOL ProcessDockInputCapture(UINT16 CaptureTime);

	/* Virtual Functions needed for Main IR Reciever RC5 (lowlevel) API to talk with Crestcore Portion */
	virtual void RC5_SetInputCaptEdge_Falling(void) {};
	virtual void RC5_SetInputCaptEdge_Rising(void) {};

	virtual void RC5_EnableInputCaptureIRQ(void) {};
	virtual void RC5_DisableInputCaptureIRQ(void) {};
	virtual void RC5_ClearInputCaptureFlag(void) {};

	virtual void RC5_EnableInputTimeoutIRQ(void) {};
	virtual void RC5_DisableInputTimeoutIRQ(void) {};
	virtual void RC5_ClearInputTimeoutFlag(void) {};

	virtual BOOL RC5_IsInputCaptureIRQFlagSet(void) {return 0;};
	virtual BOOL RC5_IsInputCaptureIRQEnabled(void) {return 0;};

	virtual BOOL RC5_IsInputTimeoutIRQFlagSet(void) {return 0;};
	virtual BOOL RC5_IsInputTimeoutIRQEnabled(void) {return 0;};

	virtual UINT16 RC5_GetCaptureTime(void) {return 0;};
	virtual void RC5_SetTimeoutValue(UINT16 CaptureTime) {};

    // class variables
	INT8   m_CodeAvailable;  //must be signec char for status
	UINT8  m_Dock;			// if true, process IR input as touchscreen dock
	UINT32 m_IrRC5Task;         //Handle for the IR Task
	UINT32 m_IrRC5Queue;		//handle for the IR Queue

};

// Function to Initialize the IR Driver Class from main loop
void IrReceiver_Init(void);
// Function Call to send out RC5 decoded code to control system via cnet or cip. Generates the final packet.
// Function body is defined in 'project'_cresnet.c file as a stub
extern void CresnetSlaveSendRF1Way( UINT32 join, UINT8 data0, UINT8 data1, UINT8 data2);

#endif  // _IRRECEIVER_H_
